#ifndef THREAD_POOL_H
#define THREAD_POOL_H

#include <thread>
#include <mutex>
#include <condition_variable>
#include <functional>
#include <future>
#include <queue>
#include <vector>
#include <memory>
#include <condition_variable>

class ThreadPool
{
public:
	ThreadPool();
	~ThreadPool();

	ThreadPool(const ThreadPool&) = delete;
	ThreadPool(ThreadPool&&) = delete;
	ThreadPool& operator=(const ThreadPool&) = delete;
	ThreadPool& operator=(ThreadPool&&) = delete;

	template<typename F, typename... Args>
	auto enqueue(F&& f, Args&&... args) -> std::future<decltype(f(args...))>
	{
		auto task = std::bind(std::forward<F>(f), std::forward<Args>(args)...);
		auto taskPtr = std::make_shared<std::packaged_task<decltype(f(args...))()>>(task);
		auto wrapper = [taskPtr]() {
			(*taskPtr)();
		};

		{
			std::lock_guard<std::mutex> lock(m_mutex);
			m_tasks.push(wrapper);
			m_condition_task.notify_one();
		}

		return taskPtr->get_future();
	}
	void wait();

private:
	mutable std::mutex m_mutex;
	std::condition_variable m_condition_task;
	std::condition_variable m_condition_finish;
	unsigned int m_busyThreads;

	std::vector<std::thread> m_threads;
	bool m_stopRequested;

	std::queue<std::function<void()>> m_tasks;

	friend class ThreadWorker;
};


class ThreadWorker
{
public:
	ThreadWorker(ThreadPool* pool);
	~ThreadWorker();

	void operator()();

private:
	ThreadPool* m_pool;
};

#endif // !THREAD_POOL_H